#!/usr/local/bin/perl
# 
# s_oc4jctl_lib.pm
# 
# Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      s_oc4jctl_lib.pm - OC4J Control Action Script OSD Library Module
#
#    DESCRIPTION
#      s_oc4jctl_lib.pm - Unix OSD library module for OC4J Control
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    jgrout      02/07/12 - Created

package s_oc4jctl_lib;

use strict;
use warnings;
use File::Spec::Functions;
use POSIX qw(:errno_h :signal_h);
use oc4jctl_common;

our $VERSION = '1';

use English;
use Exporter;

use constant {
  LOCALHOST => "localhost"
};

our (@ISA, @EXPORT);

# Export in a BEGIN block to avoid compilation failure

BEGIN {
  require Exporter;
  @ISA = qw(Exporter);

  my @exp_const = qw(LOCALHOST
                    );

  my @exp_osd  = qw(s_init_globals s_get_pid_hash s_async_start s_send_signal
                    s_java_path_defs s_get_jre s_get_jstack
                    );

  my @exp = (@exp_const, @exp_osd);
  @EXPORT  = @exp;
}

sub s_init_globals
#---------------------------------------------------------------------
# Function: Initialize Unix OSD globals
#
# Args    : None
#
# Returns : SUCC_CODE (operation succeeded)
#           FAIL_CODE (operation failed)
#---------------------------------------------------------------------
{
  return SUCC_CODE;
}

sub s_get_jre
#---------------------------------------------------------------------
# Function: Return JRE file path
#
# Args    : None
#
# Returns : JRE file path
#---------------------------------------------------------------------
{
  return catfile($ENV{"ORACLE_HOME"}, "jdk", "bin", "java");
}

sub s_get_jstack
#---------------------------------------------------------------------
# Function: Return jstack command file path
#
# Args    : None
#
# Returns : jstack command file path
#---------------------------------------------------------------------
{
  return catfile($ENV{"ORACLE_HOME"}, "jdk", "bin", "jstack");
}

sub s_get_pid_hash
#---------------------------------------------------------------------
# Function: Return a hash of Java PIDs to command lines
#
# Args    : None
#
# Returns : PID hash (key PID, value corresponding command line)
#---------------------------------------------------------------------
{
  my %pid_hash;

  my $jps = catfile($ENV{"ORACLE_HOME"}, "jdk", "bin", "jps");
  my @pid_lines = `$jps -lmv`;
  foreach my $pid_line (@pid_lines)
  {
    if ($pid_line =~ /^(\d+)\s*(.*)$/)
    {
      $pid_hash{$1} = $2;
    }
  }

  return %pid_hash;
}

sub s_async_start
#---------------------------------------------------------------------
# Function: Start command asynchronously in the background
#
# Args    : List with program name and its arguments
#
# Returns : Result of system command
#---------------------------------------------------------------------
{
  my @parms = (@_, "&");
  return system("@parms");
}

sub s_send_signal
#---------------------------------------------------------------------
# Function: Send check, quit or kill signal
#
# Args    : Type of signal to send ("check", "quit" or "kill")
#           Process number
#
# Returns : SUCC_CODE (for "quit" and "kill", operation initiated;
#                      for "check", process still under control)
#           FAIL_CODE (operation not implemented)
#           ERROR_PID (process no longer under control)
#           NULL_PID  (process no longer exists)
#---------------------------------------------------------------------
{
  my $sig_rc = FAIL_CODE;
  my ($action, $oc4j_pid) = @_;

  if ($action eq "check")
  {
    if (kill 0 => $oc4j_pid)
    {
      $sig_rc = SUCC_CODE;
    }
    elsif ($! == EPERM)
    {
      debug_out("Process $oc4j_pid no longer under our control.");
      $sig_rc = ERROR_PID;
    }
    elsif ($! == ESRCH)
    {
      debug_out("Process $oc4j_pid deceased or zombie.");
      $sig_rc = NULL_PID;
    }
    else
    {
      debug_out("Process $oc4j_pid status could not be checked.");
      $sig_rc = ERROR_PID;
    }
  }
  elsif ($action eq "quit")
  {
    kill SIGQUIT => $oc4j_pid;
    $sig_rc = SUCC_CODE;
  }
  elsif ($action eq "kill")
  {
    kill SIGKILL => $oc4j_pid;
    $sig_rc = SUCC_CODE;
  }
  return $sig_rc;
}

sub s_java_path_defs
#---------------------------------------------------------------------
# Function: Return Unix java path defs
#
# Args    : None
#
# Returns : Unix java path defs 
#---------------------------------------------------------------------
{
  return ();
}
